<?php
/**
 * lifescore functions and definitions
 *
 * @link https://developer.wordpress.org/themes/basics/theme-functions/
 *
 * @package lifescore
 */

if ( ! defined( '_S_VERSION' ) ) {
	// Replace the version number of the theme on each release.
	define( '_S_VERSION', '1.0.0' );
}

/**
 * Sets up theme defaults and registers support for various WordPress features.
 *
 * Note that this function is hooked into the after_setup_theme hook, which
 * runs before the init hook. The init hook is too late for some features, such
 * as indicating support for post thumbnails.
 */
function lifescore_setup() {
	/*
		* Make theme available for translation.
		* Translations can be filed in the /languages/ directory.
		* If you're building a theme based on lifescore, use a find and replace
		* to change 'lifescore' to the name of your theme in all the template files.
		*/
	load_theme_textdomain( 'lifescore', get_template_directory() . '/languages' );

	// Add default posts and comments RSS feed links to head.
	add_theme_support( 'automatic-feed-links' );

	/*
		* Let WordPress manage the document title.
		* By adding theme support, we declare that this theme does not use a
		* hard-coded <title> tag in the document head, and expect WordPress to
		* provide it for us.
		*/
	add_theme_support( 'title-tag' );

	/*
		* Enable support for Post Thumbnails on posts and pages.
		*
		* @link https://developer.wordpress.org/themes/functionality/featured-images-post-thumbnails/
		*/
	add_theme_support( 'post-thumbnails' );

	// This theme uses wp_nav_menu() in one location.
	register_nav_menus(
		array(
			'menu-1' => esc_html__( 'Primary', 'lifescore' ),
		)
	);

	/*
		* Switch default core markup for search form, comment form, and comments
		* to output valid HTML5.
		*/
	add_theme_support(
		'html5',
		array(
			'search-form',
			'comment-form',
			'comment-list',
			'gallery',
			'caption',
			'style',
			'script',
		)
	);

	// Set up the WordPress core custom background feature.
	add_theme_support(
		'custom-background',
		apply_filters(
			'lifescore_custom_background_args',
			array(
				'default-color' => 'ffffff',
				'default-image' => '',
			)
		)
	);

	// Add theme support for selective refresh for widgets.
	add_theme_support( 'customize-selective-refresh-widgets' );

	/**
	 * Add support for core custom logo.
	 *
	 * @link https://codex.wordpress.org/Theme_Logo
	 */
	add_theme_support(
		'custom-logo',
		array(
			'height'      => 250,
			'width'       => 250,
			'flex-width'  => true,
			'flex-height' => true,
		)
	);
}
add_action( 'after_setup_theme', 'lifescore_setup' );

/**
 * Set the content width in pixels, based on the theme's design and stylesheet.
 *
 * Priority 0 to make it available to lower priority callbacks.
 *
 * @global int $content_width
 */
function lifescore_content_width() {
	$GLOBALS['content_width'] = apply_filters( 'lifescore_content_width', 640 );
}
add_action( 'after_setup_theme', 'lifescore_content_width', 0 );

/**
 * Register widget area.
 *
 * @link https://developer.wordpress.org/themes/functionality/sidebars/#registering-a-sidebar
 */
function lifescore_widgets_init() {
	register_sidebar(
		array(
			'name'          => esc_html__( 'Sidebar', 'lifescore' ),
			'id'            => 'sidebar-1',
			'description'   => esc_html__( 'Add widgets here.', 'lifescore' ),
			'before_widget' => '<section id="%1$s" class="widget %2$s">',
			'after_widget'  => '</section>',
			'before_title'  => '<h2 class="widget-title">',
			'after_title'   => '</h2>',
		)
	);
}
add_action( 'widgets_init', 'lifescore_widgets_init' );

/**
 * Enqueue scripts and styles.
 */
function lifescore_scripts() {
	wp_enqueue_style( 'lifescore-style', get_stylesheet_uri(), array(), _S_VERSION );
	wp_style_add_data( 'lifescore-style', 'rtl', 'replace' );

	wp_enqueue_script( 'lifescore-navigation', get_template_directory_uri() . '/js/navigation.js', array(), _S_VERSION, true );

	if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
		wp_enqueue_script( 'comment-reply' );
	}
}
add_action( 'wp_enqueue_scripts', 'lifescore_scripts' );

/**
 * Implement the Custom Header feature.
 */
require get_template_directory() . '/inc/custom-header.php';

/**
 * Custom template tags for this theme.
 */
require get_template_directory() . '/inc/template-tags.php';

/**
 * Functions which enhance the theme by hooking into WordPress.
 */
require get_template_directory() . '/inc/template-functions.php';

/**
 * Customizer additions.
 */
require get_template_directory() . '/inc/customizer.php';

/**
 * Load Jetpack compatibility file.
 */
if ( defined( 'JETPACK__VERSION' ) ) {
	require get_template_directory() . '/inc/jetpack.php';
}

function wooc_extra_register_fields() {?>

	<p class="form-row form-row-first">
	<label for="reg_billing_first_name"><?php _e( 'Full name', 'woocommerce' ); ?><span class="required">*</span></label>
	<input type="text" required class="input-text" name="billing_first_name" id="reg_billing_first_name" value="<?php if ( ! empty( $_POST['billing_first_name'] ) ) esc_attr_e( $_POST['billing_first_name'] ); ?>" />
	</p>

	<div class="clear"></div>
	<?php
}
add_action( 'woocommerce_register_form_start', 'wooc_extra_register_fields' );

add_action('admin_menu', 'survey_list');

function survey_list()
{
    add_menu_page(
        'Survey Listing',     // page title
        'Survey Listing',     // menu title
        'manage_options',   // capability
        'survey-listing',     // menu slug
        'survey_list_render' // callback function
    );
}

function survey_list_render()
{
    global $title;

    print '<div class="clear"></div><div class="wrap">';
    /*print "<h1>$title</h1>";*/

    $file = plugin_dir_path( __FILE__ ) . "template_survey_list_admin.php";

    if ( file_exists( $file ) )
        require $file;

    print '</div>';
}

//Filter Code - Swatantra Bhargava - 15-05-2024
add_action('wp_ajax_filter_survey_list', 'filter_survey_list');
add_action('wp_ajax_nopriv_filter_survey_list', 'filter_survey_list');

function filter_survey_list() {

	global $wpdb;

	$user_d = $_POST['user_dtls'];

	// Make string searchable
	$user_nicename = str_replace(" ", "-", $user_d);
	$display_name = str_replace(" ", "-", $user_d);
	
	$start_date = $_POST['start_date'];
    $end_date = $_POST['end_date'];
    
	$query = "SELECT * FROM wp_survey INNER JOIN wp_users ON wp_survey.user_id = wp_users.ID WHERE 1";
	
	if(!empty($start_date) && !empty($end_date)) {
		$start_date .= " 00:00:00";
		$end_date .= " 23:59:59";
		$query .= " AND created_at >= '".$start_date."' AND created_at <= '".$end_date."'";
	}

	if(!empty($user_d)) {
		$query .= " AND (user_nicename LIKE('".$user_nicename."%') OR display_name LIKE('".$display_name."%'))";
	}

	$result = $wpdb->get_results( $query.' ORDER BY survey_id DESC ' );

    if($result) {
        foreach($result as $index => $survey){ 
			$uData = get_userdata( $survey->user_id );
			$full_name = esc_attr( $uData->first_name ) . " " . esc_attr( $uData->last_name );
			$result = $wpdb->get_results(sprintf('SELECT * FROM wp_survey WHERE survey_id=' . $survey->survey_id));
			$total_question = $result[0]->total_question * 10;


			$data = json_decode($result[0]->data);
			$cat = array();
			$chcked = array();
			$allpoint = 0;

			foreach ($data as $datas) {
				$res = explode('-', $datas);
				$cat[] = $res[0];
				$value = $res[3];
				$k = $res[0];
				$chcked[$k][] = $value;
				$allpoint = $allpoint + $value;
			}

			$percentageAll = round(($allpoint * 100 / $total_question));
			
			$survey_date = date('F j, Y', strtotime($survey->created_at));
            // Display filtered posts
            ?>
				<tr>
				   <td><span><?php echo $full_name; ?></span></td>
				   <td><span><?php echo $survey_date; ?></span></td>
				   <td><span><?php echo $percentageAll; ?>%</span></td>
				   <td>
					<ul class="action-table">
						<li> <a href="/result?surveyid=<?php echo $survey->survey_id ?>"> <i class="fa fa-eye" aria-hidden="true"></i> </a> </li>
					</ul>
				   </td>
			   </tr>
            <?php
			}
		}
    else {
        echo '<tr><td colspan="4">No record found.</td></tr>';
    }
    die();
}
function custom_static_api() {
    // Your static condition (e.g., always return true for now)
    $response = array(
        'success' => false
    );

    return rest_ensure_response($response);
}

function register_custom_static_api_route() {
    register_rest_route('custom/v1', '/status', array(
        'methods'  => 'GET',
        'callback' => 'custom_static_api',
    ));
}

add_action('rest_api_init', 'register_custom_static_api_route');

// Add custom fields to WooCommerce registration form
function custom_woocommerce_registration_fields() {
    ?>
    <p class="form-row form-row-wide">
        <label for="reg_gender"><?php _e( 'Gender', 'woocommerce' ); ?> <span class="required">*</span></label>
        <select name="gender" id="reg_gender" class="input-select" required>
            <option value=""><?php _e( 'Select a gender', 'woocommerce' ); ?></option>
            <option value="male"><?php _e( 'Male', 'woocommerce' ); ?></option>
            <option value="female"><?php _e( 'Female', 'woocommerce' ); ?></option>
            <option value="other"><?php _e( 'Other', 'woocommerce' ); ?></option>
        </select>
    </p>
    <p class="form-row form-row-wide">
        <label for="reg_age"><?php _e( 'Age', 'woocommerce' ); ?> <span class="required">*</span></label>
        <input type="number" class="input-text" name="age" id="reg_age" required maxlength="2" />
    </p>
    <?php
}
add_action( 'woocommerce_register_form_start', 'custom_woocommerce_registration_fields' );

// Validate custom fields
function custom_woocommerce_validate_fields( $username, $email, $validation_errors ) {
    if ( isset( $_POST['gender'] ) && empty( $_POST['gender'] ) ) {
        $validation_errors->add( 'gender_error', __( 'Gender is a required field.', 'woocommerce' ) );
    }
    if ( isset( $_POST['age'] ) && empty( $_POST['age'] ) ) {
        $validation_errors->add( 'age_error', __( 'Age is a required field.', 'woocommerce' ) );
    }
    return $validation_errors;
}
add_action( 'woocommerce_register_post', 'custom_woocommerce_validate_fields', 10, 3 );

// Save custom fields
function custom_woocommerce_save_fields( $customer_id ) {
    if ( isset( $_POST['gender'] ) ) {
        update_user_meta( $customer_id, 'gender', sanitize_text_field( $_POST['gender'] ) );
    }
    if ( isset( $_POST['age'] ) ) {
        update_user_meta( $customer_id, 'age', sanitize_text_field( $_POST['age'] ) );
    }
}
add_action( 'woocommerce_created_customer', 'custom_woocommerce_save_fields' );

// Add fields to edit account page
function custom_woocommerce_edit_account_fields( $user ) {
	
	$current_user = wp_get_current_user(); //get_currentuserinfo();
	$user_id = $current_user->ID;

	$all_meta_for_user = get_user_meta( $user_id );

    $gender = $all_meta_for_user["gender"][0]; //get_user_meta( $user_id, 'gender', true );
    $age = $all_meta_for_user["age"][0]; //get_user_meta( $user_id, 'age', true );
    ?>
    <p class="woocommerce-form-row woocommerce-form-row--first form-row form-row-first">
        <label for="account_gender"><?php esc_html_e( 'Gender', 'woocommerce' ); ?></label>
        <select name="account_gender" id="account_gender" class="input-select">
            <option value=""><?php _e( 'Select a gender', 'woocommerce' ); ?></option>
            <option value="male" <?php selected( $gender, 'male' ); ?>><?php _e( 'Male', 'woocommerce' ); ?></option>
            <option value="female" <?php selected( $gender, 'female' ); ?>><?php _e( 'Female', 'woocommerce' ); ?></option>
            <option value="other" <?php selected( $gender, 'other' ); ?>><?php _e( 'Other', 'woocommerce' ); ?></option>
        </select>
    </p>
    <p class="woocommerce-form-row woocommerce-form-row--last form-row form-row-last">
        <label for="account_age"><?php _e( 'Age', 'woocommerce' ); ?></label>
        <input type="number" class="input-text" name="account_age" id="account_age" value="<?php echo esc_attr( $age ); ?>" maxlength="2" />
    </p>
	<p class="woocommerce-form-row woocommerce-form-row--wide form-row form-row-wide">&nbsp;</p>
    <?php
}
add_action( 'woocommerce_edit_account_form', 'custom_woocommerce_edit_account_fields' );

// Save custom fields from edit account page
function custom_woocommerce_save_account_fields( $user_id ) {
	
    if ( isset( $_POST['account_gender'] ) ) {
        update_user_meta( $user_id, 'gender', sanitize_text_field( $_POST['account_gender'] ) );
    }
    if ( isset( $_POST['account_age'] ) ) {
        update_user_meta( $user_id, 'age', sanitize_text_field( $_POST['account_age'] ) );
    }
}
add_action( 'woocommerce_save_account_details', 'custom_woocommerce_save_account_fields' );
add_filter('wpo_force_webp_serve_using_altered_html', '__return_true');



// function create_blogs_post_type() {
//     register_post_type('blogs',
//         array(
//             'labels' => array(
//                 'name' => __('Blogs'),
//                 'singular_name' => __('Blog')
//             ),
//             'public' => true,
//             'has_archive' => true,
//             'rewrite' => array('slug' => 'blogs'),
//             'supports' => array('title', 'editor', 'thumbnail')
//         )
//     );
// }
// add_action('init', 'create_blogs_post_type');

// API WORK START

// START SIGN UP API
function custom_user_signup_api() {
    register_rest_route( 'custom/v1', '/signup/', array(
        'methods'  => 'POST',
        'callback' => 'custom_user_signup_callback',
        'permission_callback' => '__return_true', // No authentication required
    ));
}
add_action( 'rest_api_init', 'custom_user_signup_api' );

function custom_user_signup_callback( $request ) {
    $params = $request->get_params();

    // Check required fields
    if ( !isset( $params['first_name'], $params['last_name'], $params['email'], $params['password'], $params['confirm_password'] ) ) {
        return new WP_REST_Response( array(
            'status' => 'error',
            'message' => 'Missing required fields: first_name, last_name, email, password, or confirm password.'
        ), 400 );
    }

    // Sanitize inputs
    $first_name = sanitize_text_field( $params['first_name'] );
    $last_name  = sanitize_text_field( $params['last_name'] );
    $email      = sanitize_email( $params['email'] );
    $password   = sanitize_text_field( $params['password'] );
    $confirm_password = sanitize_text_field( $params['confirm_password'] );

    // Check if password and confirm password match
    if ( $password !== $confirm_password ) {
        return new WP_REST_Response( array(
            'status' => 'error',
            'message' => 'Password and confirm password do not match.'
        ), 400 );
    }

    // Check if email is valid and if the user already exists
    if ( email_exists( $email ) ) {
        return new WP_REST_Response( array(
            'status' => 'error',
            'message' => 'Email already exists.'
        ), 400 );
    }

    // Generate a username from email (if necessary, you can customize this)
    $username = explode( '@', $email )[0];

    // Create new user
    $user_id = wp_create_user( $username, $password, $email );

    if ( is_wp_error( $user_id ) ) {
        return new WP_REST_Response( array(
            'status' => 'error',
            'message' => $user_id->get_error_message()
        ), 500 );
    }

    // Add first and last name to user meta
    wp_update_user( array(
        'ID'         => $user_id,
        'first_name' => $first_name,
        'last_name'  => $last_name,
    ));
	

    return new WP_REST_Response( array(
        'status' => 'success',
        'message' => 'User registered successfully.',
        'user_id' => $user_id
    ), 200 );
}
// END ORDER STORE
// 	
// END SIGN UP API

add_action('rest_api_init', function () {
    register_rest_route('custom/v1', '/login/', array(
        'methods' => 'POST',
        'callback' => 'custom_user_login',
        'permission_callback' => '__return_true',
    ));
});
function custom_user_login($request) {
    global $wpdb;

    // Step 1: User Login
    $creds = array();
    $creds['user_login'] = sanitize_text_field($request->get_param('username'));
    $creds['user_password'] = sanitize_text_field($request->get_param('password'));
    $creds['remember'] = true;

    $user = wp_signon($creds, false);

    if (is_wp_error($user)) {
        return new WP_Error('invalid_credentials', 'Invalid username or password', array('status' => 403));
    }

    // Generate a token and save it in user meta
    $token = wp_generate_password(20, false);
    update_user_meta($user->ID, 'auth_token', $token);

    // Fetch Orders for the User
    $order_args = array(
        'customer_id' => $user->ID,
        'limit' => -1,
        'post_status' => array('processing', 'completed'),
        'orderby' => 'date',
        'order' => 'DESC',
    );
    $orders = wc_get_orders($order_args);

    if (empty($orders)) {
//         return new WP_Error('no_orders', 'No orders found for this user.', array('status' => 404));
		 $response = array(
        'user' => array(
            'id' => $user->ID,
            'username' => $user->user_login,
            'email' => $user->user_email,
            'token' => $token
        ),
			'total_survey' => isset($total_survey) ? $total_survey : 0, // Corrected syntax with ternary operator
			'current_month_survey' => isset($current_month_survey) ? (string)$current_month_survey : '0', // Corrected syntax with ternary operator
			'subscription' => array(),
			'expire_date' => 'Your Membership has expired'
		);

		return rest_ensure_response($response);         
    }

    $order = $orders[0];
// 	print_r($order->date_created);
// 	die;
    $order_date = $order->get_date_created();
	
// 	print_r($org_order_date);
// 	die;
    $has_subscription = $order->get_meta('wps_sfw_order_has_subscription');

    if ($has_subscription !== 'yes') {
		 $response = array(
			'user' => array(
				'id' => $user->ID,
				'username' => $user->user_login,
				'email' => $user->user_email,
				'token' => $token
			),
			'total_survey' => isset($total_survey) ? $total_survey : 0, // Corrected syntax with ternary operator
			'current_month_survey' => isset($current_month_survey) ? (string)$current_month_survey : '0', // Corrected syntax with ternary operator
			'subscription' => array(),  			
			'expire_date' => 'Your Membership has expired'
		);

		return rest_ensure_response($response);
		
//         return new WP_Error('no_subscription', 'Please buy a subscription first.', array('status' => 403));
    }

    // Process Items in the Order
    $items = $order->get_items();
    $attributes = array();

    foreach ($items as $item) {
        $product_id = $item->get_product_id();
        $attributes = get_post_meta($product_id);
        break; // Assuming only the first product is relevant.
    }

    if (empty($attributes)) {
        return new WP_Error('no_product_data', 'No product data found.', array('status' => 404));
    }

    $number = isset($attributes['wps_sfw_subscription_number'][0]) ? (int)$attributes['wps_sfw_subscription_number'][0] : 0;
    $interval = isset($attributes['wps_sfw_subscription_interval'][0]) ? $attributes['wps_sfw_subscription_interval'][0] : '';
    $spm = isset($attributes['survey_per_month'][0]) ? (int)$attributes['survey_per_month'][0] : 0;

    $days_to_add = $number;
    $new_date = $order_date;
	
    if ($interval === 'year') {
        $new_date = $new_date->modify("+$days_to_add years");
    } elseif ($interval === 'days') {
        $new_date = $new_date->modify("+$days_to_add days");
    } elseif ($interval === 'weeks') {
        $new_date = $new_date->modify("+$days_to_add weeks");
    } elseif ($interval === 'months') {
        $new_date = $new_date->modify("+$days_to_add months");
    }

    $current_time = current_time('timestamp');
    $expiry_time = strtotime($new_date->format('Y-m-d H:i:s'));

    if ($current_time > $expiry_time) {
//         return new WP_Error('membership_expired', 'Your membership has expired.', array('status' => 403));
		 $response = array(
			'user' => array(
				'id' => $user->ID,
				'username' => $user->user_login,
				'email' => $user->user_email,
				'token' => $token
			),
			'total_survey' => isset($total_survey) ? $total_survey : 0, // Corrected syntax with ternary operator
			'current_month_survey' => isset($current_month_survey) ? (string)$current_month_survey : '0', // Corrected syntax with ternary operator
// 			'subscription' => array(),
			'subscription' => array(
				array(
					'customer_id' => (string)$user->ID,
					'date_created' => date('Y-m-d H:i:s',strtotime($order_date)),
// 					'date_created' => date('Y-m-d H:i:s',strtotime($order_date)),
// 					'date_created' => $order->get_date_created(),
					'post_title' => $first_item ? $first_item->get_name() : 'Yearly Membership.', // Safely access the product name
					'meta_key' => 'survey_per_month',
					'meta_value' => (string)$spm
				)
			),  			
			'expire_date' => 'Your Membership has expired'
		);

		return rest_ensure_response($response);
		
// 			$response = array(
// 				'user' => array(
// 					'id' => $user->ID,
// 					'username' => $user->user_login,
// 					'email' => $user->user_email,
// 					'token' => $token
// 				)
// 			);
	
// 		$response['total_survey'] = $total_survey;
// 		$response['current_month_survey'] = $current_month_survey;
// 		$response['subscription'] = $results;
// 		// Add one year to the expiration date
// 		$newExpiryDate = date('Y-m-d', strtotime('+1 year', strtotime($expiryData)));

// 		// Get the current date
// 		$currentDate = current_time('mysql', 1); // Current time in MySQL format (e.g., '2024-11-13 12:00:00')

// 		// Check if the membership has expired (i.e., if the current date is greater than the new expiry date)
// 		if (strtotime($newExpiryDate) <= strtotime($currentDate)) {
// 			// Membership has expired
// 			$response['expire_date'] = 'Your membership has expired.';
// 		} else {
// 			// Membership is still valid
// 			$response['expire_date'] = $newExpiryDate ?? '';
// 		}
// 		return rest_ensure_response($response);
    }

    // Total Surveys Filled
    $total_survey = count($wpdb->get_results(sprintf('SELECT * FROM wp_survey WHERE user_id=%d', $user->ID)));

    // Remaining Surveys for This Month
    $month = date("m"); // Current month (01-12)
    $year = date("Y"); // Current year (YYYY)

    // Count the number of surveys for the current month and year
    $current_month_survey = $wpdb->get_var($wpdb->prepare("
        SELECT COUNT(*)
        FROM {$wpdb->prefix}survey
        WHERE user_id = %d
        AND MONTH(created_at) = %d
        AND YEAR(created_at) = %d
    ", $user->ID, $month, $year));

    // Format Response
    $response = array(
        'user' => array(
            'id' => $user->ID,
            'username' => $user->user_login,
            'email' => $user->user_email,
            'token' => $token
        ),
        'total_survey' => $total_survey,
        'current_month_survey' => (string)$current_month_survey,
//         'subscription' => array(
// //             array(
// //                 'customer_id' => (string)$user->ID,
// //                 'date_created' => $order_date->format('Y-m-d H:i:s'),
// //                 'post_title' => $order->get_items()[0]->get_name() : 'No product', // Assuming the first item name is the title
// //                 'meta_key' => 'survey_per_month',
// //                 'meta_value' => (string)$spm
// //             )
//         ),
'subscription' => array(
        array(
            'customer_id' => (string)$user->ID,
            'date_created' => $order_date->format('Y-m-d H:i:s'),
            'post_title' => $first_item ? $first_item->get_name() : 'Yearly Membership.', // Safely access the product name
            'meta_key' => 'survey_per_month',
            'meta_value' => (string)$spm
        )
    ),         
        'expire_date' => $new_date->format('Y-m-d')
    );

    return rest_ensure_response($response);
}


// function custom_user_login($request) {
//     global $wpdb;

//     // Step 1: User Login
//     $creds = array();
//     $creds['user_login'] = sanitize_text_field($request->get_param('username'));
//     $creds['user_password'] = sanitize_text_field($request->get_param('password'));
//     $creds['remember'] = true;

//     $user = wp_signon($creds, false);

//     if (is_wp_error($user)) {
//         return new WP_Error('invalid_credentials', 'Invalid username or password', array('status' => 403));
//     }

//     // Generate a token and save it in user meta
//     $token = wp_generate_password(20, false);
//     update_user_meta($user->ID, 'auth_token', $token);

//     // Fetch Orders for the User
//     $order_args = array(
//         'customer_id' => $user->ID,
//         'limit' => -1,
//         'post_status' => array('processing', 'completed'),
//         'orderby' => 'date',
//         'order' => 'DESC',
//     );
//     $orders = wc_get_orders($order_args);

//     if (empty($orders)) {
//         return new WP_Error('no_orders', 'No orders found for this user.', array('status' => 404));
//     }

//     $order = $orders[0];
//     $order_date = $order->get_date_created();
//     $has_subscription = $order->get_meta('wps_sfw_order_has_subscription');

//     if ($has_subscription !== 'yes') {
//         return new WP_Error('no_subscription', 'Please buy a subscription first.', array('status' => 403));
//     }

//     // Process Items in the Order
//     $items = $order->get_items();
//     $attributes = array();

//     foreach ($items as $item) {
//         $product_id = $item->get_product_id();
//         $attributes = get_post_meta($product_id);
//         break; // Assuming only the first product is relevant.
//     }

//     if (empty($attributes)) {
//         return new WP_Error('no_product_data', 'No product data found.', array('status' => 404));
//     }

//     $number = isset($attributes['wps_sfw_subscription_number'][0]) ? (int)$attributes['wps_sfw_subscription_number'][0] : 0;
//     $interval = isset($attributes['wps_sfw_subscription_interval'][0]) ? $attributes['wps_sfw_subscription_interval'][0] : '';
//     $spm = isset($attributes['survey_per_month'][0]) ? (int)$attributes['survey_per_month'][0] : 0;

//     $days_to_add = $number;
//     $new_date = $order_date;

//     if ($interval === 'year') {
//         $new_date = $order_date->modify("+$days_to_add years");
//     } elseif ($interval === 'days') {
//         $new_date = $order_date->modify("+$days_to_add days");
//     } elseif ($interval === 'weeks') {
//         $new_date = $order_date->modify("+$days_to_add weeks");
//     } elseif ($interval === 'months') {
//         $new_date = $order_date->modify("+$days_to_add months");
//     }

//     $current_time = current_time('timestamp');
//     $expiry_time = strtotime($new_date->format('Y-m-d H:i:s'));

//     if ($current_time > $expiry_time) {
//         return new WP_Error('membership_expired', 'Your membership has expired.', array('status' => 403));
//     }

//     // Check Survey Limits
//     $total_surveys_taken = 0; // Replace with actual logic to count surveys taken.
//     if ($total_surveys_taken >= $spm) {
//         return new WP_Error('survey_limit_exceeded', 'Your per month survey limit has been exceeded.', array('status' => 403));
//     }

// 	return rest_ensure_response(array(
// 		'message' => 'Login successful.',
// 		'membership_expiry' => $new_date->format('Y-m-d H:i:s'),
// 		'remaining_surveys' => $spm - $total_surveys_taken,
// 		'user' => array(
// 			'id' => $user->ID,
// 			'username' => $user->user_login,
// 			'email' => $user->user_email,
// 			'token' => $token
// 		)
// 	));

// }

// function custom_user_login($request) {
//     global $wpdb;

//     // Step 1: User Login
//     $creds = array();
//     $creds['user_login'] = sanitize_text_field($request->get_param('username'));
//     $creds['user_password'] = sanitize_text_field($request->get_param('password'));
//     $creds['remember'] = true;

//     $user = wp_signon($creds, false);

//     if (is_wp_error($user)) {
//         return new WP_Error('invalid_credentials', 'Invalid username or password', array('status' => 403));
//     }

//     // Generate a token and save it in user meta
//     $token = wp_generate_password(20, false);
//     update_user_meta($user->ID, 'auth_token', $token);

//     // Fetch Orders for the User
//     $order_args = array(
//         'customer_id' => $user->ID,
//         'limit' => -1,
//         'post_status' => array('processing', 'completed'),
//         'orderby' => 'date',
//         'order' => 'DESC',
//     );
//     $orders = wc_get_orders($order_args);

//     if (empty($orders)) {
//         return new WP_Error('no_orders', 'No orders found for this user.', array('status' => 404));
//     }

//     $order = $orders[0];
//     $order_date = $order->get_date_created();
//     $has_subscription = $order->get_meta('wps_sfw_order_has_subscription');

//     if ($has_subscription !== 'yes') {
//         return new WP_Error('no_subscription', 'Please buy a subscription first.', array('status' => 403));
//     }

//     // Process Items in the Order
//     $items = $order->get_items();
//     $attributes = array();

//     foreach ($items as $item) {
//         $product_id = $item->get_product_id();
//         $attributes = get_post_meta($product_id);
//         break; // Assuming only the first product is relevant.
//     }

//     if (empty($attributes)) {
//         return new WP_Error('no_product_data', 'No product data found.', array('status' => 404));
//     }

//     $number = isset($attributes['wps_sfw_subscription_number'][0]) ? (int)$attributes['wps_sfw_subscription_number'][0] : 0;
//     $interval = isset($attributes['wps_sfw_subscription_interval'][0]) ? $attributes['wps_sfw_subscription_interval'][0] : '';
//     $spm = isset($attributes['survey_per_month'][0]) ? (int)$attributes['survey_per_month'][0] : 0;

//     $days_to_add = $number;
//     $new_date = $order_date;

//     if ($interval === 'year') {
//         $new_date = $order_date->modify("+$days_to_add years");
//     } elseif ($interval === 'days') {
//         $new_date = $order_date->modify("+$days_to_add days");
//     } elseif ($interval === 'weeks') {
//         $new_date = $order_date->modify("+$days_to_add weeks");
//     } elseif ($interval === 'months') {
//         $new_date = $order_date->modify("+$days_to_add months");
//     }

//     $current_time = current_time('timestamp');
//     $expiry_time = strtotime($new_date->format('Y-m-d H:i:s'));

//     if ($current_time > $expiry_time) {
//         return new WP_Error('membership_expired', 'Your membership has expired.', array('status' => 403));
//     }

//     // Total Surveys Filled
//     $total_survey = count($wpdb->get_results(sprintf('SELECT * FROM wp_survey WHERE user_id=%d', $user->ID)));

//     // Remaining Surveys for This Month
//     $month = date("m"); // Current month (01-12)
//     $year = date("Y"); // Current year (YYYY)

//     // Count the number of surveys for the current month and year
//     $current_month_survey = $wpdb->get_var($wpdb->prepare("
//         SELECT COUNT(*)
//         FROM {$wpdb->prefix}survey
//         WHERE user_id = %d
//         AND MONTH(created_at) = %d
//         AND YEAR(created_at) = %d
//     ", $user->ID, $month, $year));

//     // Check Survey Limits
//     if ($current_month_survey >= $spm) {
//         return new WP_Error('survey_limit_exceeded', 'Your per month survey limit has been exceeded.', array('status' => 403));
//     }

//     return rest_ensure_response(array(
//         'message' => 'Login successful.',
//         'membership_expiry' => $new_date->format('Y-m-d H:i:s'),
//         'remaining_surveys' => $spm - $current_month_survey,
//         'total_surveys_filled' => $total_survey,
//         'user' => array(
//             'id' => $user->ID,
//             'username' => $user->user_login,
//             'email' => $user->user_email,
//             'token' => $token
//         )
//     ));
// }



// 
// START LOGOUT CODE
add_action('rest_api_init', function () {
    register_rest_route('custom/v1', '/logout/', array(
        'methods' => 'POST',
        'callback' => 'custom_user_logout',
        'permission_callback' => '__return_true',
    ));
});

function custom_user_logout($request) {
    // Get the user ID from the token
    $token = $request->get_param('token');

    // Validate token
    $user = get_users(array(
        'meta_key' => 'auth_token',
        'meta_value' => $token,
        'number' => 1,
        'fields' => 'ID'
    ));

    if (empty($user)) {
        return new WP_Error('invalid_token', 'Invalid token', array('status' => 403));
    }

    // Log out the user
    wp_logout();

    // Clear the token to prevent reuse
    delete_user_meta($user[0], 'auth_token');
    
    // Optionally clear all auth cookies
    wp_clear_auth_cookie();

    return rest_ensure_response(array(
        'message' => 'User successfully logged out'
    ));
}

// END LOGOUT CODE
// 
// Add custom REST API endpoint for 'Forgot Password'
add_action('rest_api_init', function () {
    register_rest_route('custom/v1', '/reset-password', array(
        'methods' => 'POST',
        'callback' => 'custom_reset_password', // Renamed function
        'permission_callback' => '__return_true', // Handle permissions properly in production
    ));
});

function custom_reset_password(WP_REST_Request $request)
{
    $email = sanitize_email($request->get_param('email'));
	$new_password = wp_generate_password( 12, true, false );
	if ( empty( $email ) ) {
		$json = array( 'code' => '0', 'msg' => 'Please enter login user detail' );
		echo json_encode( $json );
		exit;     
	}

	$userdata = get_user_by( 'email', $email); 

	if ( empty( $userdata ) ) {
		$userdata = get_user_by( 'login', $$email );
	}

	if ( empty( $userdata ) ) {
        return new WP_REST_Response(['message' => 'Email does not exist in our system.'], 200);
	}

	$user = new WP_User( intval( $userdata->ID ) ); 
	
 	$admin_email = get_option('admin_email'); // Admin email to send the contact form
     $subject = " Reset Password Request - My Life Advice " ;
     $email_message .= "Email: $email\n\n";
	 $email_message .= "New Password: $new_password\n\n";
	 $message = "We have received a request to reset your password. Please use the new password to login.\n\n";
     $email_message .= "Message:\n$message\n";
     $headers = ['Content-Type: text/plain; charset=UTF-8', 'Reply-To: ' . $admin_email];
//     // Send the email
     	$mail_sent = wp_mail($email, $subject, $email_message, $headers);

     if ($mail_sent) {
		 wp_set_password($new_password, $user->ID);

         return new WP_REST_Response(['message' => 'Check your email password has been sended successfully.'], 200);
     } else {
         return new WP_REST_Response(['message' => 'Failed to send the message. Please try again later.'], 500);
     }	
}
 
// 
// START UPDATE ACCOUNT API
add_action('rest_api_init', function () {
    register_rest_route('custom/v1', '/update-account/', array(
        'methods' => 'POST',
        'callback' => 'update_user_account',
        'permission_callback' => '__return_true', // Token validation is handled in the callback
    ));
});

function update_user_account($request) {
    // Get the user ID from the request if provided
    $user_id = isset($request['user_id']) ? intval($request['user_id']) : 0;

    // If no user ID, use the token to find the user
    if (!$user_id) {
        $token = isset($request['token']) ? sanitize_text_field($request['token']) : '';

        // Validate the token and get the user ID
        $user = get_users(array(
            'meta_key' => 'auth_token',
            'meta_value' => $token,
            'number' => 1,
            'fields' => 'ID'
        ));

        if (empty($user)) {
            return new WP_Error('invalid_token', 'Invalid or missing token', array('status' => 403));
        }

        $user_id = $user[0]; // Get the user ID from the array
    }

    // Ensure the user exists
    if (!get_userdata($user_id)) {
        return new WP_Error('invalid_user', 'User does not exist', array('status' => 404));
    }

    // Get fields to update from the request
    $first_name = isset($request['first_name']) ? sanitize_text_field($request['first_name']) : '';
    $last_name = isset($request['last_name']) ? sanitize_text_field($request['last_name']) : '';
    $email = isset($request['email']) ? sanitize_email($request['email']) : '';
    $age = isset($request['age']) ? intval($request['age']) : '';
    $gender = isset($request['gender']) ? sanitize_text_field($request['gender']) : '';

    // Password and Confirm Password
    $password = isset($request['password']) ? sanitize_text_field($request['password']) : '';
    $confirm_password = isset($request['confirm_password']) ? sanitize_text_field($request['confirm_password']) : '';

    // Prepare an array of fields to update
    $userdata = array(
        'ID' => $user_id,
    );

    if (!empty($first_name)) {
        $userdata['first_name'] = $first_name;
    }

    if (!empty($last_name)) {
        $userdata['last_name'] = $last_name;
    }

    if (!empty($email)) {
        // Ensure email is valid and unique
        if (!is_email($email)) {
            return new WP_Error('invalid_email', 'Invalid email address', array('status' => 400));
        }

        if (email_exists($email) && email_exists($email) != $user_id) {
            return new WP_Error('email_exists', 'This email address is already registered', array('status' => 400));
        }

        $userdata['user_email'] = $email;
    }

    // Check if password and confirm password are provided and match
    if (!empty($password) && !empty($confirm_password)) {
        if ($password !== $confirm_password) {
            return new WP_Error('password_mismatch', 'Password and confirm password do not match', array('status' => 400));
        }

        // Update the password
        $userdata['user_pass'] = $password;
    } else {
        // Password or confirm password is not provided; skip password update
        if (!empty($password) || !empty($confirm_password)) {
            return new WP_Error('incomplete_password', 'Both password and confirm password must be provided to change the password', array('status' => 400));
        }
    }

    // Update user information
    $updated_user_id = wp_update_user($userdata);

    if (is_wp_error($updated_user_id)) {
        return new WP_Error('update_failed', 'Failed to update user account', array('status' => 500));
    }

    // Update age and gender as user meta
    if (!empty($age)) {
        update_user_meta($user_id, 'age', $age);
    }

    if (!empty($gender)) {
        update_user_meta($user_id, 'gender', $gender);
    }

    return rest_ensure_response(array('message' => 'User account updated successfully.'));
}
//
// 
// START USER DETAILS API
add_action('rest_api_init', function () {
    register_rest_route('custom/v1', '/get-user-details/', array(
        'methods' => 'POST',
        'callback' => 'get_user_details',
        'permission_callback' => '__return_true', // Token validation is handled in the callback
    ));
});

function get_user_details($request) {
    // Get the user ID or token from the request
    $user_id = isset($request['user_id']) ? intval($request['user_id']) : 0;
    $token = isset($request['token']) ? sanitize_text_field($request['token']) : '';

    // If no user ID is provided, use the token to find the user
    if (!$user_id && !empty($token)) {
        $user = get_users(array(
            'meta_key' => 'auth_token',
            'meta_value' => $token,
            'number' => 1,
            'fields' => 'ID'
        ));

        if (empty($user)) {
            return new WP_Error('invalid_token', 'Invalid or missing token', array('status' => 403));
        }

        $user_id = $user[0]; // Get the user ID from the array
    }

    // Ensure the user exists
    if (!$user_id || !get_userdata($user_id)) {
        return new WP_Error('invalid_user', 'User does not exist', array('status' => 404));
    }

    // Get the user data
    $user_data = get_userdata($user_id);
    $first_name = get_user_meta($user_id, 'first_name', true);
    $last_name = get_user_meta($user_id, 'last_name', true);
    $age = get_user_meta($user_id, 'age', true);
    $gender = get_user_meta($user_id, 'gender', true);

    // Prepare response data
    $response_data = array(
        'ID' => $user_data->ID,
        'user_login' => $user_data->user_login,
        'user_email' => $user_data->user_email,
        'first_name' => $first_name,
        'last_name' => $last_name,
        'age' => $age,
        'gender' => $gender,
    );

    return rest_ensure_response($response_data);
}

// END USER DETAILS API
//   
// START DELETE ACCOUNT API
add_action('rest_api_init', function () {
    register_rest_route('custom/v1', '/delete-account/', array(
        'methods' => 'POST',
        'callback' => 'delete_user_account',
        'permission_callback' => '__return_true', // Token validation is handled in the callback
    ));
});

function delete_user_account($request) {
    // Get the token from the request body
    $token = isset($request['token']) ? sanitize_text_field($request['token']) : '';

    // Validate the token and get the user ID
    $user = get_users(array(
        'meta_key' => 'auth_token',
        'meta_value' => $token,
        'number' => 1,
        'fields' => 'ID'
    ));

    if (empty($user)) {
        return new WP_Error('invalid_token', 'Invalid or missing token', array('status' => 403));
    }

    $user_id = $user[0]; // Get the user ID from the array

    // Ensure the user exists
    if (!get_userdata($user_id)) {
        return new WP_Error('invalid_user', 'User does not exist', array('status' => 404));
    }

    // Delete the user account
    $deleted = wp_delete_user($user_id);

    if ($deleted) {
        return rest_ensure_response(array('message' => 'User account deleted successfully.'));
    } else {
        return new WP_Error('delete_failed', 'Failed to delete user account', array('status' => 500));
    }
}

// END DELETE ACCOUNT API
// 
// Articles Lists
add_action('rest_api_init', function () {
    register_rest_route('custom/v1', '/articles/', array(
        'methods' => 'POST',
        'callback' => 'get_articles_posts_with_token',
        'permission_callback' => '__return_true', // Token validation is handled inside the callback
    ));
});

function get_articles_posts_with_token($request) {
    // Get token from the request body
    $token = isset($request['token']) ? sanitize_text_field($request['token']) : '';

    // Validate the token
    $user = get_users(array(
        'meta_key' => 'auth_token',
        'meta_value' => $token,
        'number' => 1,
        'fields' => 'ID'
    ));

    if (empty($user)) {
        return new WP_Error('invalid_token', 'Invalid or missing token', array('status' => 403));
    }

    // If token is valid, fetch articles
    $args = array(
        'post_type' => 'articles',
        'posts_per_page' => -1, // Fetch all articles
        'post_status' => 'publish',
    );

    $query = new WP_Query($args);

    if ($query->have_posts()) {
        $articles = array();

        while ($query->have_posts()) {
            $query->the_post();
            
            $articles[] = array(
                'id' => get_the_ID(),
                'title' => get_the_title(),
                'content' => get_the_content(),
                'excerpt' => get_the_excerpt(),
                'date' => get_the_date(),
                'author' => get_the_author(),
                'featured_image' => get_the_post_thumbnail_url(get_the_ID(), 'full'),
            );
        }

        wp_reset_postdata();

        return rest_ensure_response($articles);
    } else {
        return rest_ensure_response(array('message' => 'No articles found.'));
    }
}

// // END ARTICLES LISTS

// Single Article Records
add_action('rest_api_init', function () {
    register_rest_route('custom/v1', '/article/', array(
        'methods' => 'POST',
        'callback' => 'get_single_article_with_post_method',
        'permission_callback' => '__return_true', // No authentication for endpoint itself, handled in callback
    ));
});

function get_single_article_with_post_method($request) {
    // Get token from the request body
    $token = isset($request['token']) ? sanitize_text_field($request['token']) : '';

    // Validate the token
    $user = get_users(array(
        'meta_key' => 'auth_token',
        'meta_value' => $token,
        'number' => 1,
        'fields' => 'ID'
    ));

    if (empty($user)) {
        return new WP_Error('invalid_token', 'Invalid or missing token', array('status' => 403));
    }

    // Get the article ID from the request body
    $article_id = isset($request['id']) ? intval($request['id']) : 0;

    // Validate article ID
    if ($article_id <= 0) {
        return new WP_Error('invalid_id', 'Invalid article ID', array('status' => 400));
    }

    // Get the post by ID
    $post = get_post($article_id);

    // Check if the post exists and is from the 'articles' post type
    if (!$post || $post->post_type !== 'articles' || $post->post_status !== 'publish') {
        return new WP_Error('not_found', 'Article not found', array('status' => 404));
    }

    // Prepare the response data
    $response = array(
        'id' => $post->ID,
        'title' => $post->post_title,
        'content' => $post->post_content,
        'excerpt' => $post->post_excerpt,
        'date' => get_the_date('', $post->ID),
        'author' => get_the_author_meta('display_name', $post->post_author),
        'featured_image' => get_the_post_thumbnail_url($post->ID, 'full'),
    );

    return rest_ensure_response($response);
}

// START ALL PRODUCTS
add_action('rest_api_init', function () {
    register_rest_route('custom/v1', '/products/', array(
        'methods' => 'POST',
        'callback' => 'get_product_posts_with_token',
        'permission_callback' => '__return_true', // Token validation inside callback
    ));
});

function get_product_posts_with_token($request) {
    // Get the token from the request body
    $token = isset($request['token']) ? sanitize_text_field($request['token']) : '';

    // Check if the token is provided
    if (empty($token)) {
        return new WP_Error('missing_token', 'Token is required', array('status' => 403));
    }

    // Validate the token
    $user = get_users(array(
        'meta_key' => 'auth_token',
        'meta_value' => $token,
        'number' => 1,
        'fields' => 'ID'
    ));

    if (empty($user)) {
        return new WP_Error('invalid_token', 'Invalid or missing token', array('status' => 403));
    }

    // If token is valid, fetch the products
    $args = array(
        'post_type' => 'product',
        'posts_per_page' => -1, // Fetch all products
        'post_status' => 'publish',
    );

    $query = new WP_Query($args);

    if ($query->have_posts()) {
        $products = array();

        while ($query->have_posts()) {
            $query->the_post();

            // Collect product data
            $products[] = array(
                'id' => get_the_ID(),
                'title' => get_the_title(),
                'content' => get_the_content(),
                'excerpt' => get_the_excerpt(),
                'price' => get_post_meta(get_the_ID(), '_price', true), // Example price meta field
                'featured_image' => get_the_post_thumbnail_url(get_the_ID(), 'full'),
                'date' => get_the_date(),
                'author' => get_the_author(),
            );
        }

        wp_reset_postdata();

        return rest_ensure_response($products);
    } else {
        return rest_ensure_response(array('message' => 'No products found.'));
    }
}

// END ALL PRODUCTS
// 
// START SURVEY Options RECORDS
add_action('rest_api_init', function () {
    register_rest_route('custom/v1', '/taxonomy/survey-section/', array(
        'methods' => 'POST',
        'callback' => 'get_survey_section_terms_with_token',
        'permission_callback' => '__return_true', // Token validation inside callback
    ));
});

function get_survey_section_terms_with_token($request) {
    // Get the token from the request body
    $token = isset($request['token']) ? sanitize_text_field($request['token']) : '';

    // Check if the token is provided
    if (empty($token)) {
        return new WP_Error('missing_token', 'Token is required', array('status' => 403));
    }

    // Validate the token
    $user = get_users(array(
        'meta_key' => 'auth_token',
        'meta_value' => $token,
        'number' => 1,
        'fields' => 'ID'
    ));

    if (empty($user)) {
        return new WP_Error('invalid_token', 'Invalid or missing token', array('status' => 403));
    }

    // Fetch terms for the survey-section taxonomy
    $terms = get_terms(array(
        'taxonomy' => 'survey-section',
        'hide_empty' => true, // Set to false if you want to show empty terms
    ));

    if (is_wp_error($terms)) {
        return new WP_Error('term_error', 'Error fetching terms', array('status' => 500));
    }

    return rest_ensure_response($terms);
}

// END SURVEY Options RECORDS

// START SURVEY RECORDS
add_action('rest_api_init', function () {
    register_rest_route('custom/v1', '/surveys/', array(
        'methods' => 'POST',
        'callback' => 'get_survey_records_with_token',
        'permission_callback' => '__return_true', // Token validation inside callback
    ));
});

function get_survey_records_with_token($request) {
    // Get the token from the request body
    $token = isset($request['token']) ? sanitize_text_field($request['token']) : '';

    // Check if the token is provided
    if (empty($token)) {
        return new WP_Error('missing_token', 'Token is required', array('status' => 403));
    }

    // Validate the token
    $user = get_users(array(
        'meta_key' => 'auth_token',
        'meta_value' => $token,
        'number' => 1,
        'fields' => 'ID'
    ));

    if (empty($user)) {
        return new WP_Error('invalid_token', 'Invalid or missing token', array('status' => 403));
    }

// Fetch survey records
$args = array(
    'post_type' => 'survey', // Custom post type for surveys
    'posts_per_page' => -1, // Fetch all surveys
    'post_status' => 'publish',
);

$query = new WP_Query($args);

if ($query->have_posts()) {
    $surveys = array();

    while ($query->have_posts()) {
        $query->the_post();

        // Get categories
        $categories = wp_get_post_terms(get_the_ID(), 'category', array('fields' => 'names'));
        
        // Get survey sections
//         $sections = wp_get_post_terms(get_the_ID(), 'survey-section', array('fields' => 'names'));
        // Get survey sections with IDs
        $sections_terms = wp_get_post_terms(get_the_ID(), 'survey-section'); // Get term objects
        
        // Prepare sections data
        $sections = array();
        if (!is_wp_error($sections_terms) && !empty($sections_terms)) {
            foreach ($sections_terms as $term) {
                $sections[] = array(
                    'id' => $term->term_id,  // Term ID
                    'name' => $term->name,    // Term name
                );
            }
        }         

        // Prepare survey data for the response
        $surveys[] = array(
            'id' => get_the_ID(),
            'title' => get_the_title(),
            'sections' => $sections, // Include survey sections
        );
    }

    wp_reset_postdata();
    return rest_ensure_response($surveys);
} else {
    return new WP_Error('not_found', 'No surveys found', array('status' => 404));
}

}
// END SURVEY RECORDS
// 
// START SINGLE SURVEY API
// START Survey LIST Fetch By SURVEY ID
// Function to get surveys by survey ID
function get_surveys_by_survey_id_post(WP_REST_Request $request) {
    // Get the token from the request body
    $token = isset($request['token']) ? sanitize_text_field($request['token']) : '';

    // Check if the token is provided
    if (empty($token)) {
        return new WP_Error('missing_token', 'Token is required', array('status' => 403));
    }

    // Validate the token by checking against the user meta
    $user = get_users(array(
        'meta_key' => 'auth_token',
        'meta_value' => $token,
        'number' => 1,
        'fields' => 'ID'
    ));

    if (empty($user)) {
        return new WP_Error('invalid_token', 'Invalid or missing token', array('status' => 403));
    }

    global $wpdb;
    $tablename = $wpdb->prefix . 'survey';

    // Get survey_id from the request body
    $survey_id = $request->get_param('survey_id');

    // Check if survey_id is provided
    if (empty($survey_id)) {
        return new WP_REST_Response([
            'status' => 'error',
            'message' => 'Survey ID is required'
        ], 400);
    }

    // Retrieve surveys for the specified survey_id
    $surveys = $wpdb->get_results($wpdb->prepare(
        "SELECT * FROM $tablename WHERE survey_id = %d", 
        $survey_id
    ), ARRAY_A); // Use ARRAY_A to return results as an associative array

    // Check if surveys were found
    if (empty($surveys)) {
        return new WP_REST_Response([
            'status' => 'error',
            'message' => 'No surveys found for this survey ID'
        ], 404);
    }

    // Prepare the response data
    $formatted_surveys = [];
    foreach ($surveys as $survey) {
        $formatted_surveys[] = [
            'survey_id' => (int) $survey['survey_id'], // Adding the survey_id
            'user_id' => (int) $survey['user_id'],
            'cat' => json_decode($survey['data'], true), // Assuming 'data' is a JSON string
            'question_data' => json_decode($survey['question_data'], true), // Assuming 'question_data' is a JSON string
            'date' => $survey['created_at']
        ];
    }

    // Return a successful response with the formatted surveys
    return new WP_REST_Response([
        'status' => 'success',
        'surveys' => $formatted_surveys
    ], 200);
}

// Register the REST API endpoint
add_action('rest_api_init', function () {
    register_rest_route('custom/v1', '/survey/details', array(
        'methods' => 'POST',
        'callback' => 'get_surveys_by_survey_id_post',
        'permission_callback' => '__return_true', // Adjust permissions as needed
    ));
});

// END SINGLE SURVEY API
// START FORUMS API
add_action('rest_api_init', function () {
    register_rest_route('custom/v1', '/forums/', array(
        'methods' => 'POST',
        'callback' => 'get_forum_records_with_token',
        'permission_callback' => '__return_true', // Token validation inside callback
    ));
});
function get_forum_records_with_token(WP_REST_Request $request) {
    // Get the token from the request body
    $token = isset($request['token']) ? sanitize_text_field($request['token']) : '';

    // Check if the token is provided
    if (empty($token)) {
        return new WP_Error('missing_token', 'Token is required', array('status' => 403));
    }

    // Validate the token
    $user = get_users(array(
        'meta_key' => 'auth_token',
        'meta_value' => $token,
        'number' => 1,
        'fields' => 'ID'
    ));

    if (empty($user)) {
        return new WP_Error('invalid_token', 'Invalid or missing token', array('status' => 403));
    }
    // Fetch all forums first
    $args = array(
        'post_type' => 'forum',
        'post_status' => 'publish',
        'numberposts' => -1 // Get all forums
    );

    $forums = get_posts($args);

    // Prepare the response
    $forums_data = [];
    foreach ($forums as $forum) {
        // For each forum, fetch its topics (assuming 'topic' is the post type for topics)
        $topics_args = array(
            'post_type' => 'topic',
            'post_status' => 'publish',
            'numberposts' => -1,
            'post_parent' => $forum->ID // Get topics belonging to this forum
        );

        $topics = get_posts($topics_args);

        // Prepare the topics data
        $topics_data = [];
        foreach ($topics as $topic) {
            // For each topic, fetch its replies (assuming 'reply' is the post type for replies)
            $replies_args = array(
                'post_type' => 'reply',
                'post_status' => 'publish',
                'numberposts' => -1,
                'post_parent' => $topic->ID // Get replies belonging to this topic
            );

            $replies = get_posts($replies_args);

            // Prepare the replies data
            $replies_data = [];
            foreach ($replies as $reply) {
                $replies_data[] = array(
                    'id' => $reply->ID,
                    'content' => $reply->post_content,
                    'date' => $reply->post_date,
                    'author' => get_the_author_meta('display_name', $reply->post_author),
                );
            }

            // Prepare the topic data, including replies
            $topics_data[] = array(
                'id' => $topic->ID,
                'title' => $topic->post_title,
                'content' => $topic->post_content,
                'date' => $topic->post_date,
                'author' => get_the_author_meta('display_name', $topic->post_author),
                'replies' => $replies_data // Include the replies for this topic
            );
        }

        // Add the forum and its topics (including replies) to the response
        $forums_data[] = array(
            'id' => $forum->ID,
            'title' => $forum->post_title,
            'content' => $forum->post_content,
            'date' => $forum->post_date,
            'author' => get_the_author_meta('display_name', $forum->post_author),
            'topics' => $topics_data // Include the topics and their replies for this forum
        );
    }

    return rest_ensure_response($forums_data);
}
// END FORUMS API
// 
// START SINGLE FORM API
// 
// Register the REST API routes
add_action('rest_api_init', function () {
    // Endpoint for getting a single forum (with POST method) without an ID parameter
    register_rest_route('custom/v1', '/forums-details', array(
        'methods' => 'POST',
        'callback' => 'get_single_forum_with_token',
        'permission_callback' => '__return_true', // Token validation inside callback
    ));
});

// Callback function to get a single forum
function get_single_forum_with_token(WP_REST_Request $request) {
    $token = isset($request['token']) ? sanitize_text_field($request['token']) : '';
    $forum_id = isset($request['forum_id']) ? intval($request['forum_id']) : null; // Get forum ID from request body

    if (empty($token)) {
        return new WP_Error('missing_token', 'Token is required', array('status' => 403));
    }

    $user = get_users(array(
        'meta_key' => 'auth_token',
        'meta_value' => $token,
        'number' => 1,
        'fields' => 'ID'
    ));

    if (empty($user)) {
        return new WP_Error('invalid_token', 'Invalid or missing token', array('status' => 403));
    }

    if (is_null($forum_id)) {
        return new WP_Error('missing_forum_id', 'Forum ID is required', array('status' => 400));
    }

    // Fetch the specific forum by ID
    $forum = get_post($forum_id);

    if (!$forum || $forum->post_type !== 'forum' || $forum->post_status !== 'publish') {
        return new WP_Error('not_found', 'Forum not found', array('status' => 404));
    }

    // Prepare the response for the single forum
    $topics_args = array(
        'post_type' => 'topic',
        'post_status' => 'publish',
        'numberposts' => -1,
        'post_parent' => $forum->ID // Get topics belonging to this forum
    );

    $topics = get_posts($topics_args);
    $topics_data = [];

    foreach ($topics as $topic) {
        $replies_args = array(
            'post_type' => 'reply',
            'post_status' => 'publish',
            'numberposts' => -1,
            'post_parent' => $topic->ID // Get replies belonging to this topic
        );

        $replies = get_posts($replies_args);
        $replies_data = [];

        foreach ($replies as $reply) {
            $replies_data[] = array(
                'id' => $reply->ID,
                'content' => $reply->post_content,
                'date' => $reply->post_date,
                'author' => get_the_author_meta('display_name', $reply->post_author),
            );
        }

        $topics_data[] = array(
            'id' => $topic->ID,
            'title' => $topic->post_title,
            'content' => $topic->post_content,
            'date' => $topic->post_date,
            'author' => get_the_author_meta('display_name', $topic->post_author),
            'replies' => $replies_data // Include the replies for this topic
        );
    }

    // Prepare the response for the forum
    $forum_data = array(
        'id' => $forum->ID,
        'title' => $forum->post_title,
        'content' => $forum->post_content,
        'date' => $forum->post_date,
		'data' => true,
        'author' => get_the_author_meta('display_name', $forum->post_author),
        'topics' => $topics_data // Include the topics and their replies for this forum
    );

    return rest_ensure_response($forum_data);
}

// 
// END SINGLE FORUM API
// Register the REST API route for submitting replies
add_action('rest_api_init', function () {
    register_rest_route('custom-api/v1', '/submit-reply/', array(
        'methods' => 'POST',
        'callback' => 'submit_topic_reply',
        'permission_callback' => '__return_true', // Adjust permissions as needed
    ));
});

// Function to handle reply submission
function submit_topic_reply($request) {
	
	    $token = isset($request['token']) ? sanitize_text_field($request['token']) : '';

    // Check if the token is provided
    if (empty($token)) {
        return new WP_Error('missing_token', 'Token is required', array('status' => 403));
    }

    // Validate the token
    $user = get_users(array(
        'meta_key' => 'auth_token',
        'meta_value' => $token,
        'number' => 1,
        'fields' => 'ID'
    ));

    if (empty($user)) {
        return new WP_Error('invalid_token', 'Invalid or missing token', array('status' => 403));
    }
// 	echo $user[0];
// 	die;
    // Get the necessary parameters from the request
    $topic_id = $request->get_param('topic_id');
    $forum_id = $request->get_param('forum_id');
    $reply_content = $request->get_param('reply_content');
    $user_id = $user[0]; // Assuming you are using the logged-in user

    // Validate inputs
    if (!$topic_id || !$forum_id || !$reply_content) {
        return new WP_Error('missing_data', 'Missing required data', array('status' => 400));
    }

    // Check if the topic belongs to the specified forum (post_parent of the topic should match the forum ID)
    $topic = get_post($topic_id);
    if (!$topic || $topic->post_type !== 'topic' || $topic->post_parent != $forum_id) {
        return new WP_Error('invalid_topic', 'Invalid topic or forum', array('status' => 400));
    }

    // Insert the reply as a new post of type 'reply'
    $reply_id = wp_insert_post(array(
        'post_type' => 'reply',
        'post_status' => 'publish',
        'post_content' => $reply_content,
        'post_title' => 'Reply to ' . $topic->post_title,
        'post_parent' => $topic_id, // Associate the reply with the specific topic
        'post_author' => $user_id,
    ));

    // Check if the reply was successfully added
    if (is_wp_error($reply_id)) {
        return new WP_Error('reply_failed', 'Failed to add reply', array('status' => 500));
    }

    // Return success response
    return rest_ensure_response(array(
        'success' => true,
        'message' => 'Reply added successfully',
        'reply_id' => $reply_id,
    ));
}

// END SINGLE FORUM API
// 
// START WORK ON FORM REPLY
// add_action('rest_api_init', function () {
//     register_rest_route('custom/v1', '/reply/', array(
//         'methods' => 'POST',
//         'callback' => 'submit_forum_reply',
//         'permission_callback' => '__return_true', // Adjust permissions as needed
//     ));
// });
// function submit_forum_reply(WP_REST_Request $request) {
//     // Get the parameters
//     $post_id = $request->get_param('post_id'); // ID of the forum topic
//     $reply_content = $request->get_param('reply_content'); // Content of the reply
//     $token = $request->get_param('token'); // Authentication token (if necessary)

//     // Validate parameters (you can add more validation)
//     if (empty($post_id) || empty($reply_content)) {
//         return new WP_Error('missing_data', 'Post ID and reply content are required.', array('status' => 400));
//     }

//     // Assuming user authentication and token validation here, if needed
//     // (You can include your token validation logic if required)

//     // Create a new reply (you might need to adjust this based on your forum plugin)
//     $reply_data = array(
//         'post_content' => $reply_content,
//         'post_parent' => $post_id, // This indicates it's a reply to a specific post
//         'post_type' => 'reply', // Ensure this matches your reply post type
//         'post_status' => 'publish', // Publish the reply immediately
//         'post_author' => get_current_user_id(), // Set the current user as the author
//     );

//     $reply_id = wp_insert_post($reply_data);

//     if (is_wp_error($reply_id)) {
//         return new WP_Error('reply_failed', 'Failed to submit reply.', array('status' => 500));
//     }

//     // Return success response
//     return rest_ensure_response(array('message' => 'Reply submitted successfully', 'reply_id' => $reply_id));
// }

// END WORK ON FORUM REPLY
// 
// START Coupon API
add_action('rest_api_init', function () {
    register_rest_route('custom/v1', '/coupons/', array(
        'methods' => 'POST',
        'callback' => 'get_coupon_records_with_token',
        'permission_callback' => '__return_true', // Token validation inside callback
    ));
});

function get_coupon_records_with_token($request) {
    // Get the token from the request body
    $token = isset($request['token']) ? sanitize_text_field($request['token']) : '';

    // Check if the token is provided
    if (empty($token)) {
        return new WP_Error('missing_token', 'Token is required', array('status' => 403));
    }

    // Validate the token
    $user = get_users(array(
        'meta_key' => 'auth_token',
        'meta_value' => $token,
        'number' => 1,
        'fields' => 'ID'
    ));

    if (empty($user)) {
        return new WP_Error('invalid_token', 'Invalid or missing token', array('status' => 403));
    }

    // Fetch coupon records
    $args = array(
        'post_type' => 'shop_coupon',
        'posts_per_page' => -1, // Fetch all coupons
        'post_status' => 'publish',
    );

    $query = new WP_Query($args);

    if ($query->have_posts()) {
        $coupons = array();

        while ($query->have_posts()) {
            $query->the_post();

            // Prepare coupon data for the response
            $coupons[] = array(
                'id' => get_the_ID(),
                'code' => get_the_title(),
                'amount' => get_post_meta(get_the_ID(), 'coupon_amount', true),
                'discount_type' => get_post_meta(get_the_ID(), 'discount_type', true),
                'date_expires' => get_post_meta(get_the_ID(), 'date_expires', true),
                'usage_count' => get_post_meta(get_the_ID(), 'usage_count', true),
                'description' => get_the_content(),
                'date_created' => get_the_date(),
                'author' => get_the_author(),
            );
        }

        wp_reset_postdata();
        return rest_ensure_response($coupons);
    } else {
        return new WP_Error('not_found', 'No coupons found', array('status' => 404));
    }
}
// END Coupon API
// 
// START SURVEY STORE API
add_action('rest_api_init', function () {
    register_rest_route('custom/v1', '/survey-submit', array(
        'methods' => 'POST',
        'callback' => 'submit_survey_data',
        'permission_callback' => '__return_true',  // If you want to restrict, change this as needed
    ));
});
// function submit_survey_data(WP_REST_Request $request) {
//     // Get the token from the request body
//     global $wpdb;
//     $token = isset($request['token']) ? sanitize_text_field($request['token']) : '';

//     // Check if the token is provided
//     if (empty($token)) {
//         return new WP_Error('missing_token', 'Token is required', array('status' => 403));
//     }

//     // Validate the token
//     $user = get_users(array(
//         'meta_key' => 'auth_token',
//         'meta_value' => $token,
//         'number' => 1,
//         'fields' => 'ID'
//     ));

//     if (empty($user)) {
//         return new WP_Error('invalid_token', 'Invalid or missing token', array('status' => 403));
//     }
// $userid = $user[0];

// // Prepare the SQL query
// $query = $wpdb->prepare('SELECT COUNT(*) as total FROM wp_survey WHERE user_id = %d AND created_at > NOW() - INTERVAL 1 MONTH', $userid);
// $result = $wpdb->get_results($query);
// // Initialize $total
// $total = !empty($result) ? $result[0]->total : 0;

// $orderArg = array(
//     'customer_id' => $userid,
//     'limit' => -1,
//     'post_status' => array('processing', 'completed'),
//     'orderby' => 'date',
//     'order' => 'DESC',
// );

// $orders = wc_get_orders($orderArg);
// if (empty($orders)) {
// 	return new WP_Error('invalid_membership', 'Please Buy Membership first', array('status' => 403));
// }
	
//     // Get parameters from the request
//     $cat_data = $request->get_param('cat');
//     $question_data = $request->get_param('question_data');

//     if (empty($cat_data) || empty($question_data)) {
//         return new WP_REST_Response([
//             'status' => 'error',
//             'message' => 'Missing data fields'
//         ], 400);
//     }

//     // Encode the category data into JSON
//     $data = json_encode($cat_data);
//     $total_questions = count($cat_data);

//     // Extract specific questions based on keys
//     $selectedQuestions = [26, 29, 30];
//     $orgArr = array();
//     foreach ($selectedQuestions as $key) {
//         if (isset($question_data[$key])) {
//             $orgArr[$key] = $question_data[$key];
//         }
//     }
//     // Encode the selected question data into JSON
//     $question_data_json = json_encode($orgArr);

//     // Insert into the database
//     $tablename = $wpdb->prefix . 'survey';
//     $sql = $wpdb->prepare(
//         "INSERT INTO $tablename (user_id, total_question, data, question_data) VALUES (%d, %d, %s, %s)",
//         $userid, $total_questions, $data, $question_data_json
//     );
//     $wpdb->query($sql);

//     // Get the last inserted ID
//     $lastid = $wpdb->insert_id;

//     // Return a successful response
//     return new WP_REST_Response([
//         'status' => 'success',
//         'message' => 'Survey data inserted successfully',
//         'survey_id' => $lastid
//     ], 200);
// }
function submit_survey_data(WP_REST_Request $request) {
    global $wpdb;

    // Get the token from the request body
    $token = isset($request['token']) ? sanitize_text_field($request['token']) : '';

    // Check if the token is provided
    if (empty($token)) {
        return new WP_Error('missing_token', 'Token is required', array('status' => 403));
    }

    // Validate the token
    $user = get_users(array(
        'meta_key' => 'auth_token',
        'meta_value' => $token,
        'number' => 1,
        'fields' => 'ID'
    ));

    if (empty($user)) {
        return new WP_Error('invalid_token', 'Invalid or missing token', array('status' => 403));
    }
    $userid = $user[0];

    // Prepare the SQL query to check survey submissions
    $query = $wpdb->prepare('SELECT COUNT(*) as total FROM wp_survey WHERE user_id = %d AND created_at > NOW() - INTERVAL 1 MONTH', $userid);
    $result = $wpdb->get_results($query);
    $total = !empty($result) ? $result[0]->total : 0;

    // Get user's latest orders
    $orderArg = array(
        'customer_id' => $userid,
        'limit' => -1,
        'post_status' => array('processing', 'completed'),
        'orderby' => 'date',
        'order' => 'DESC',
    );

    $orders = wc_get_orders($orderArg);
    if (empty($orders)) {
        return new WP_Error('invalid_membership', 'Please Buy Membership first', array('status' => 403));
    }
    $items = $orders[0]->get_items();

    // Get product attributes from the first order item
    foreach ($items as $item) {
        $pid = $item->get_product_id();
        break; // Get only the first item
    }

    $attributes = get_post_meta($pid);
    $number = isset($attributes['wps_sfw_subscription_number'][0]) ? $attributes['wps_sfw_subscription_number'][0] : 0;
    $interval = isset($attributes['wps_sfw_subscription_interval'][0]) ? $attributes['wps_sfw_subscription_interval'][0] : '';
    $spm = isset($attributes['survey_per_month'][0]) ? $attributes['survey_per_month'][0] : 0;

    // Calculate new expiration date
    $tdays = (1 * $number);
    $newdata = '';

    switch ($interval) {
        case 'year':
            $newdata = date('Y-m-d H:i:s', strtotime("+$tdays year"));
            break;
        case 'days':
            $newdata = date('Y-m-d H:i:s', strtotime("+$tdays days"));
            break;
        case 'weeks':
            $newdata = date('Y-m-d H:i:s', strtotime("+$tdays weeks"));
            break;
        case 'months':
            $newdata = date('Y-m-d H:i:s', strtotime("+$tdays months"));
            break;
    }

    $date_now = current_time('mysql', 1); // Get current time in MySQL format
    $mdate = strtotime($newdata);

    // Check membership status
    $etype = 0; // Default value
    if (strtotime($date_now) > $mdate) {
        $etype = 1; // Membership expired
    }
    if ($total >= $spm) {
        $etype = 2; // Per month limit exceeded
    }

    if (($date_now < $newdata) && ($total < $spm)) {
        // Get parameters from the request
        $cat_data = $request->get_param('cat');
        $question_data = $request->get_param('question_data');

        if (empty($cat_data) || empty($question_data)) {
            return new WP_REST_Response(['status' => 'error', 'message' => 'Missing data fields'], 400);
        }

        // Encode the category data into JSON
        $data = json_encode($cat_data);
        $total_questions = count($cat_data);

        // Extract specific questions based on keys
        $selectedQuestions = [26, 29, 30];
        $orgArr = [];
        foreach ($selectedQuestions as $key) {
            if (isset($question_data[$key])) {
                $orgArr[$key] = $question_data[$key];
            }
        }
        // Encode the selected question data into JSON
        $question_data_json = json_encode($orgArr);

        // Insert into the database
        $tablename = $wpdb->prefix . 'survey';
        $sql = $wpdb->prepare("INSERT INTO $tablename (user_id, total_question, data, question_data) VALUES (%d, %d, %s, %s)", $userid, $total_questions, $data, $question_data_json);
        $wpdb->query($sql);

        // Get the last inserted ID
        $lastid = $wpdb->insert_id;

        // Return a successful response
        return new WP_REST_Response(['status' => 'success', 'message' => 'Survey data inserted successfully', 'survey_id' => $lastid], 200);
    } else {
        // Handle cases for expired membership or exceeded limits
        if ($etype == 2) {
            return new WP_REST_Response(['status' => 'error', 'message' => 'Your Per Month Limit Exceeded'], 403);
        }
        if ($etype == 1) {
            return new WP_REST_Response(['status' => 'error', 'message' => 'Your Membership Expired'], 403);
        }
    }
}

// function submit_survey_data(WP_REST_Request $request) {
	
//     // Get the token from the request body
//     $token = isset($request['token']) ? sanitize_text_field($request['token']) : '';

//     // Check if the token is provided
//     if (empty($token)) {
//         return new WP_Error('missing_token', 'Token is required', array('status' => 403));
//     }

//     // Validate the token
//     $user = get_users(array(
//         'meta_key' => 'auth_token',
//         'meta_value' => $token,
//         'number' => 1,
//         'fields' => 'ID'
//     ));

//     if (empty($user)) {
//         return new WP_Error('invalid_token', 'Invalid or missing token', array('status' => 403));
//     }	
	
	
//     // Get parameters from the request
//     $user_id = $request->get_param('user_id');
//     $cat_data = $request->get_param('cat');
//     $question_data = $request->get_param('question_data');

//     if (empty($cat_data) || empty($question_data)) {
//         return new WP_REST_Response([
//             'status' => 'error',
//             'message' => 'Missing data fields'
//         ], 400);
//     }

// //     Encode the category data into JSON
//     $data = json_encode($cat_data);
// //     Count the total number of 'cat' items
//     $total = count($cat_data);

// //     Extract the specific questions based on keys
//     $selectedQuestions = [26, 29, 30];
//     $orgArr = array();
//     foreach ($selectedQuestions as $key) {
//         if (isset($question_data[$key])) {
//             $orgArr[$key] = $question_data[$key];
//         }
//     }
// //     Encode the selected question data into JSON
//     $question_data_json = json_encode($orgArr);

// //     Use WordPress $wpdb to interact with the database
//     global $wpdb;
//     $tablename = $wpdb->prefix . 'survey';

//     // Insert into the database
//     $sql = $wpdb->prepare(
//         "INSERT INTO $tablename (user_id, total_question, data, question_data) VALUES (%d, %d, %s, %s)",
//         $user_id, $total, $data, $question_data_json
//     );
//     $wpdb->query($sql);

// //     Get the last inserted ID
//     $lastid = $wpdb->insert_id;

//     // Return a successful response
//     return new WP_REST_Response([
//         'status' => 'success',
//         'message' => 'Survey data inserted successfully',
//         'survey_id' => $lastid
//     ], 200);
// }
// END SURVEY STORE API
// 
// START SURVEY UPDATE RECORD
add_action('rest_api_init', function () {
    register_rest_route('custom/v1', '/survey-update', array(
        'methods' => 'POST',
        'callback' => 'update_survey_data',
        'permission_callback' => '__return_true',  // Token handled in the callback
    ));
});

function update_survey_data(WP_REST_Request $request) {

    // Get the token from the request body
    $token = isset($request['token']) ? sanitize_text_field($request['token']) : '';

    // Check if the token is provided
    if (empty($token)) {
        return new WP_Error('missing_token', 'Token is required', array('status' => 403));
    }

    // Validate the token
    $user = get_users(array(
        'meta_key' => 'auth_token',
        'meta_value' => $token,
        'number' => 1,
        'fields' => 'ID'
    ));

    if (empty($user)) {
        return new WP_Error('invalid_token', 'Invalid or missing token', array('status' => 403));
    }   

    // Get parameters from the request
    $user_id = $request->get_param('user_id');
    $cat_data = $request->get_param('cat');
    $question_data = $request->get_param('question_data');
    $survey_id = $request->get_param('survey_id'); // The ID of the survey to update

    if (empty($cat_data) || empty($question_data) || empty($survey_id)) {
        return new WP_REST_Response([
            'status' => 'error',
            'message' => 'Missing data fields'
        ], 400);
    }

    // Encode the category data into JSON
    $data = json_encode($cat_data);
    // Count the total number of 'cat' items
    $total = count($cat_data);

    // Extract the specific questions based on keys
    $selectedQuestions = [26, 29, 30];
    $orgArr = array();
    foreach ($selectedQuestions as $key) {
        if (isset($question_data[$key])) {
            $orgArr[$key] = $question_data[$key];
        }
    }
    // Encode the selected question data into JSON
    $question_data_json = json_encode($orgArr);

    // Use WordPress $wpdb to interact with the database
    global $wpdb;
    $tablename = $wpdb->prefix . 'survey';

    // Update the database with new survey data
    $sql = $wpdb->prepare(
        "UPDATE $tablename 
         SET user_id = %d, total_question = %d, data = %s, question_data = %s 
         WHERE survey_id = %d", // `id` is the survey_id in this case
        $user_id, $total, $data, $question_data_json, $survey_id
    );
    
    $update_result = $wpdb->query($sql);

    // Check if the update was successful
    if ($update_result === false) {
        return new WP_REST_Response([
            'status' => 'error',
            'message' => 'Error updating the survey data'
        ], 500);
    }

    // Return a successful response
    return new WP_REST_Response([
        'status' => 'success',
        'message' => 'Survey data updated successfully',
        'survey_id' => $survey_id
    ], 200);
}


// END SURVEY UPDATE RECORD
// 
// START Survey LIST Fetch By USER ID
// Function to get surveys by user ID
function get_surveys_by_user_id_post(WP_REST_Request $request) {
    // Get the token from the request body
    $token = isset($request['token']) ? sanitize_text_field($request['token']) : '';

    // Check if the token is provided
    if (empty($token)) {
        return new WP_Error('missing_token', 'Token is required', array('status' => 403));
    }

    // Validate the token
    $user = get_users(array(
        'meta_key' => 'auth_token',
        'meta_value' => $token,
        'number' => 1,
        'fields' => 'ID'
    ));

    if (empty($user)) {
        return new WP_Error('invalid_token', 'Invalid or missing token', array('status' => 403));
    }   	
    global $wpdb;
    $tablename = $wpdb->prefix . 'survey';

    // Get user_id from the request body
    $user_id = $request->get_param('user_id');

    // Check if user_id is provided
    if (empty($user_id)) {
        return new WP_REST_Response([
            'status' => 'error',
            'message' => 'User ID is required'
        ], 400);
    }

    // Retrieve surveys for the specified user_id
    $surveys = $wpdb->get_results($wpdb->prepare(
        "SELECT * FROM $tablename WHERE user_id = %d", 
        $user_id
    ), ARRAY_A); // Use ARRAY_A to return results as an associative array

    // Check if surveys were found
    if (empty($surveys)) {
        return new WP_REST_Response([
            'status' => 'error',
            'message' => 'No surveys found for this user'
        ], 404);
    }

    // Prepare the response data
    $formatted_surveys = [];
    foreach ($surveys as $survey) {
        $formatted_surveys[] = [
            'survey_id' => (int) $survey['survey_id'], // Adding the survey_id
			'user_id' => (int) $survey['user_id'],
            'cat' => json_decode($survey['data'], true), // Assuming 'data' is a JSON string
            'question_data' => json_decode($survey['question_data'], true), // Assuming 'question_data' is a JSON string
			'date' => $survey['created_at']
        ];
    }

    // Return a successful response with the formatted surveys
    return new WP_REST_Response([
        'status' => 'success',
        'surveys' => $formatted_surveys
    ], 200);
}

// Register the REST API endpoint
add_action('rest_api_init', function () {
    register_rest_route('custom/v1', '/surveys/user', array(
        'methods' => 'POST',
        'callback' => 'get_surveys_by_user_id_post',
        'permission_callback' => '__return_true', // Adjust permissions as needed
    ));
});

// END SURVEY LIST FETCH BY USER ID
// 
// 
// SURVEY START RESULT API
// 
add_action('rest_api_init', function () {
    register_rest_route('custom/v1', '/survey-result/', array(
        'methods' => 'POST',
        'callback' => 'get_survey_result',
        'permission_callback' => '__return_true', // Allow public access
    ));
});
function get_survey_result(WP_REST_Request $request) {
    global $wpdb;

    // Get the token, survey ID, and user ID from the request body
    $token = isset($request['token']) ? sanitize_text_field($request['token']) : '';
    $user_id = isset($request['user_id']) ? intval($request['user_id']) : '';
    $survey_id = $request->get_param('surveyid');

    // Check if the token is provided
    if (empty($token)) {
        return new WP_Error('missing_token', 'Token is required', array('status' => 403));
    }

    // Validate the token
    $user = get_users(array(
        'meta_key' => 'auth_token',
        'meta_value' => $token,
        'number' => 1,
        'fields' => 'ID'
    ));

    if (empty($user)) {
        return new WP_Error('invalid_token', 'Invalid or missing token', array('status' => 403));
    }

    // Fetch survey data
    $query = $wpdb->prepare(
        'SELECT * FROM wp_survey WHERE survey_id = %d AND user_id = %d  ORDER BY survey_id DESC',
        $survey_id,
        $user_id
    );
    $result = $wpdb->get_results($query);

    if (empty($result)) {
        return new WP_Error('no_survey', 'Survey not found', array('status' => 404));
    }

    // Initialize variables
    $total_questions = $result[0]->total_question;
    $total_points = $total_questions * 10;
    $data = json_decode($result[0]->data);
    $categories = array();
    $checked = array();
    $total_score = 0;

    // Process the survey data
    foreach ($data as $entry) {
        $res = explode('-', $entry);
        $category = $res[0];
        $value = (int)$res[3];
        
        // Store the category and values
        $categories[] = $category;
        $checked[$category][] = $value;
        $total_score += $value;
    }

    // Calculate overall percentage
    $overall_percentage = round(($total_score * 100) / $total_points);

    // Process each category and calculate its percentage and color
    $category_results = array();
    foreach ($checked as $category => $values) {
        $subtotal = count($values) * 10;
        $sum = array_sum($values);
        $category_percentage = round(($sum * 100) / $subtotal);

        if ($category_percentage > 100) {
            $category_percentage -= 100;
        }

        // Determine color based on percentage range
        $color = '#FF0000'; // Default color
        if ($category_percentage >= 91  && $percentageAll <= 100) {
            $color = '#008000';
        } elseif ($category_percentage >= 81 && $percentageAll <= 90) {
            $color = '#008000';
        } elseif ($category_percentage >= 71 && $percentageAll <= 80) {
            $color = '#008000';
        } elseif ($category_percentage >= 61 && $percentageAll <= 70) {
            $color = '#FFFF00';
        } elseif ($category_percentage >= 51 && $percentageAll <= 60) {
            $color = '#FFA500';
        } elseif ($category_percentage >= 41 && $percentageAll <= 50) {
            $color = '#FF0000';
        } elseif ($category_percentage >= 31 && $percentageAll <= 40) {
            $color = '#FF0000';
        } elseif ($category_percentage >= 21 && $percentageAll <= 30) {
            $color = '#FF0000';
        } elseif ($category_percentage >= 11 && $percentageAll <= 20) {
            $color = '#FF0000';
        }

        // Store category percentage and color
        $category_results[] = array(
            'category' => $category,
            'percentage' => $category_percentage,
            'color' => $color
        );
    }

    // Average score with the same gender and age
    $survey_user_id = $result[0]->user_id;
    $user_age = get_user_meta($survey_user_id, 'age', true); 
    $user_gender = get_user_meta($survey_user_id, 'gender', true); 

    $overallargs = array(
        'meta_query' => array(
            array(
                'key' => 'age',
                'value' => $user_age,
                'compare' => '='
            ),
            array(
                'key' => 'gender',
                'value' => $user_gender,
                'compare' => '='
            )
        )
    );

    $member_arr = get_users($overallargs);
    $sameageusers = array();
    $finalOverallScore = 0;

    if ($member_arr) {
        foreach ($member_arr as $user) {
            $sameageusers[] = $user->ID;
        }
        
        $userids = implode(",", $sameageusers);
        $overallsurverys = $wpdb->get_results(sprintf('SELECT * FROM wp_survey WHERE user_id IN (' . $userids . ')'));
        
        $percentageAll = array();
        $countofSurveys = 0;

        foreach ($overallsurverys as $overallresult) {
            $overalltquestion = $overallresult->total_question;
            $overalltotal = $overalltquestion * 10;
            $overalldata = json_decode($overallresult->data);
            $overallallpoint = 0;

            foreach ($overalldata as $overalldatas) {
                $overallres = explode('-', $overalldatas);
                $overallvalue = $overallres[3];
                $overallallpoint += (int)$overallvalue;
            }

            $percentageAll[] = round(($overallallpoint * 100 / $overalltotal));
            $countofSurveys++;
        }

        if ($countofSurveys > 0) {
            $finalOverallScore = array_sum($percentageAll) / $countofSurveys;
        }
    }

    // Average score without filtering by gender and age
    $wintoutmember_arr = get_users();
    $wintoutsameageusers = array();
    $wintoutfinalOverallScore = 0;

    if ($wintoutmember_arr) {
        foreach ($wintoutmember_arr as $wintoutuser) {
            $wintoutsameageusers[] = $wintoutuser->ID;
        }
        
        $wintoutuserids = implode(",", $wintoutsameageusers);
        $wintoutoverallsurverys = $wpdb->get_results(sprintf('SELECT * FROM wp_survey WHERE user_id IN (' . $wintoutuserids . ')'));
        
        $wintoutpercentageAll = array();
        $wintoutcountofSurveys = 0;

        foreach ($wintoutoverallsurverys as $wintoutoverallresult) {
            $wintoutoveralltquestion = $wintoutoverallresult->total_question;
            $wintoutoveralltotal = $wintoutoveralltquestion * 10;
            $wintoutoveralldata = json_decode($wintoutoverallresult->data);
            $wintoutoverallallpoint = 0;

            foreach ($wintoutoveralldata as $wintoutoveralldatas) {
                $wintoutoverallres = explode('-', $wintoutoveralldatas);
                $wintoutoverallvalue = $wintoutoverallres[3];
                $wintoutoverallallpoint += (int)$wintoutoverallvalue;
            }

            $wintoutpercentageAll[] = round(($wintoutoverallallpoint * 100 / $wintoutoveralltotal));
            $wintoutcountofSurveys++;
        }

        if ($wintoutcountofSurveys > 0) {
            $wintoutfinalOverallScore = array_sum($wintoutpercentageAll) / $wintoutcountofSurveys;
        }
    }
// Determine color based on percentage range
function getColorBasedOnPercentage($percentage) {
    if ($percentage >= 91 && $percentage <= 100) {
        return '#008000'; // Green
    } elseif ($percentage >= 81 && $percentage <= 90) {
        return '#008000'; // Green
    } elseif ($percentage >= 71 && $percentage <= 80) {
        return '#008000'; // Green
    } elseif ($percentage >= 61 && $percentage <= 70) {
        return '#FFFF00'; // Yellow
    } elseif ($percentage >= 51 && $percentage <= 60) {
        return '#FFA500'; // Orange
    } elseif ($percentage >= 41 && $percentage <= 50) {
        return '#FF0000'; // Red
    } elseif ($percentage >= 31 && $percentage <= 40) {
        return '#FF0000'; // Red
    } elseif ($percentage >= 21 && $percentage <= 30) {
        return '#FF0000'; // Red
    } elseif ($percentage >= 11 && $percentage <= 20) {
        return '#FF0000'; // Red
    } else {
        return '#FF0000'; // Default Red
    }
}

// Average score with the same gender and age (logic remains the same)
// ...

// Calculate color for final overall score based on same gender and age
$finalOverallScoreColor = getColorBasedOnPercentage($finalOverallScore);

// Average score without filtering by gender and age (logic remains the same)
// ...

// Calculate color for final overall score without gender and age filter
$wintoutfinalOverallScoreColor = getColorBasedOnPercentage($wintoutfinalOverallScore);
	
	$overall_percentageColor = getColorBasedOnPercentage($overall_percentage);
    // Return the result as a JSON response
    return array(
        'user_id' => $user_id, // Include user_id in the response
        'survey_id' => $survey_id,
        'categories' => $category_results,
		'overall_percentage' => round($overall_percentage),
		'overall_percentageColor' => $overall_percentageColor,    
		'final_overall_score_same_gender_age' => round($finalOverallScore),
		'final_overall_score_same_gender_age_color' => $finalOverallScoreColor, // Color based on percentage
		'final_overall_score_all' => round($wintoutfinalOverallScore),
		'final_overall_score_all_color' => $wintoutfinalOverallScoreColor
	);
}

// function get_survey_result(WP_REST_Request $request) {
//     global $wpdb;

//     // Get the token, survey ID, and user ID from the request body
//     $token = isset($request['token']) ? sanitize_text_field($request['token']) : '';
//     $user_id = isset($request['user_id']) ? intval($request['user_id']) : '';
//     $survey_id = $request->get_param('surveyid');

//     // Check if the token is provided
//     if (empty($token)) {
//         return new WP_Error('missing_token', 'Token is required', array('status' => 403));
//     }

//     // Validate the token
//     $user = get_users(array(
//         'meta_key' => 'auth_token',
//         'meta_value' => $token,
//         'number' => 1,
//         'fields' => 'ID'
//     ));

//     if (empty($user)) {
//         return new WP_Error('invalid_token', 'Invalid or missing token', array('status' => 403));
//     }

//     // Fetch survey data
//     $query = $wpdb->prepare(
//         'SELECT * FROM wp_survey WHERE survey_id = %d AND user_id = %d',
//         $survey_id,
//         $user_id
//     );
//     $result = $wpdb->get_results($query);

//     if (empty($result)) {
//         return new WP_Error('no_survey', 'Survey not found', array('status' => 404));
//     }

//     // Initialize variables
//     $total_questions = $result[0]->total_question;
//     $total_points = $total_questions * 10;
//     $data = json_decode($result[0]->data);
//     $categories = array();
//     $checked = array();
//     $total_score = 0;

//     // Process the survey data
//     foreach ($data as $entry) {
//         $res = explode('-', $entry);
//         $category = $res[0];
//         $value = (int)$res[3];
        
//         // Store the category and values
//         $categories[] = $category;
//         $checked[$category][] = $value;
//         $total_score += $value;
//     }

//     // Calculate overall percentage
//     $overall_percentage = round(($total_score * 100) / $total_points);

//     // Process each category and calculate its percentage and color
//     $category_results = array();
//     foreach ($checked as $category => $values) {
//         $subtotal = count($values) * 10;
//         $sum = array_sum($values);
//         $category_percentage = round(($sum * 100) / $subtotal);

//         if ($category_percentage > 100) {
//             $category_percentage -= 100;
//         }

//         // Determine color based on percentage range
//         if ($category_percentage >= 91 && $category_percentage <= 100) {
//             $color = 'darkblue';
//         } elseif ($category_percentage >= 81 && $category_percentage <= 90) {
//             $color = 'white';
//         } elseif ($category_percentage >= 71 && $category_percentage <= 80) {
//             $color = 'black';
//         } elseif ($category_percentage >= 61 && $category_percentage <= 70) {
//             $color = 'red';
//         } elseif ($category_percentage >= 51 && $category_percentage <= 60) {
//             $color = 'skyblue';
//         } elseif ($category_percentage >= 41 && $category_percentage <= 50) {
//             $color = 'purple';
//         } elseif ($category_percentage >= 31 && $category_percentage <= 40) {
//             $color = 'green';
//         } elseif ($category_percentage >= 21 && $category_percentage <= 30) {
//             $color = 'sky';
//         } elseif ($category_percentage >= 11 && $category_percentage <= 20) {
//             $color = 'yellow';
//         } else {
//             $color = 'blue';
//         }

//         // Store category percentage and color
//         $category_results[] = array(
//             'category' => $category,
//             'percentage' => $category_percentage,
//             'color' => $color
//         );
//     }

//     // Return the result as a JSON response
//     return array(
//         'user_id' => $user_id, // Include user_id in the response
//         'survey_id' => $survey_id,
//         'overall_percentage' => $overall_percentage,
//         'categories' => $category_results
//     );
// }
// 
// SURVEY END RESULT API
// 
// 
// START FORMS GETS TYPE LIST
// 
add_action('rest_api_init', function () {
    register_rest_route('custom/v1', '/revisions', array(
        'methods' => 'GET',
        'callback' => 'get_forum_topics',
        'permission_callback' => '__return_true', // Adjust permissions as needed
    ));
});

function get_forum_topics() {
    // Fetch all forums first
    $args = array(
        'post_type' => 'forum',
        'post_status' => 'publish',
        'numberposts' => -1 // Get all forums
    );

    $forums = get_posts($args);

    // Prepare the response
    $forums_data = [];
    foreach ($forums as $forum) {
        // For each forum, fetch its topics (assuming 'topic' is the post type for topics)
        $topics_args = array(
            'post_type' => 'topic',
            'post_status' => 'publish',
            'numberposts' => -1,
            'post_parent' => $forum->ID // Get topics belonging to this forum
        );

        $topics = get_posts($topics_args);

        // Prepare the topics data
        $topics_data = [];
        foreach ($topics as $topic) {
            // For each topic, fetch its replies (assuming 'reply' is the post type for replies)
            $replies_args = array(
                'post_type' => 'reply',
                'post_status' => 'publish',
                'numberposts' => -1,
                'post_parent' => $topic->ID // Get replies belonging to this topic
            );

            $replies = get_posts($replies_args);

            // Prepare the replies data
            $replies_data = [];
            foreach ($replies as $reply) {
                $replies_data[] = array(
                    'id' => $reply->ID,
                    'content' => $reply->post_content,
                    'date' => $reply->post_date,
                    'author' => get_the_author_meta('display_name', $reply->post_author),
                );
            }

            // Prepare the topic data, including replies
            $topics_data[] = array(
                'id' => $topic->ID,
                'title' => $topic->post_title,
                'content' => $topic->post_content,
                'date' => $topic->post_date,
                'author' => get_the_author_meta('display_name', $topic->post_author),
                'replies' => $replies_data // Include the replies for this topic
            );
        }

        // Add the forum and its topics (including replies) to the response
        $forums_data[] = array(
            'id' => $forum->ID,
            'title' => $forum->post_title,
            'content' => $forum->post_content,
            'date' => $forum->post_date,
            'author' => get_the_author_meta('display_name', $forum->post_author),
            'topics' => $topics_data // Include the topics and their replies for this forum
        );
    }

    return rest_ensure_response($forums_data);
}




// 
// END FORM FORUMS GETS TYPE LIST
// 
// START FORUM TOPIC GET
// 
add_action('rest_api_init', function () {
    register_rest_route('custom/v1', '/topics/(?P<parent_id>\d+)', array(
        'methods' => 'GET',
        'callback' => 'get_topics_by_parent',
        'permission_callback' => '__return_true', // Adjust permissions as needed
    ));
});

function get_topics_by_parent($data) {
    $parent_id = intval($data['parent_id']); // Get the parent ID from the request

    // Fetch all posts of the 'topic' post type with the specified parent ID
    $args = array(
        'post_type' => 'topic',
        'post_parent' => $parent_id,
        'post_status' => 'publish',
        'numberposts' => -1 // Get all posts
    );

    $topics = get_posts($args);

    // Prepare the response
    $topics_data = [];
    foreach ($topics as $topic) {
        $topics_data[] = array(
            'id' => $topic->ID,
            'title' => $topic->post_title,
            'content' => $topic->post_content,
            'date' => $topic->post_date,
            'author' => get_the_author_meta('display_name', $topic->post_author),
        );
    }

    return rest_ensure_response($topics_data);
}
// END FORUM TOPIC GETS
// 
// 
// START TOPIC REPLY
// 
add_action('rest_api_init', function () {
    register_rest_route('custom/v1', '/create-reply/', array(
        'methods' => 'POST',
        'callback' => 'create_reply',
        'permission_callback' => '__return_true', // Adjust permissions as needed
    ));
});

function create_reply(WP_REST_Request $request) {
    // Get the post parent and content from the request
    $post_parent = intval($request->get_param('post_parent')); 
    $post_content = sanitize_textarea_field($request->get_param('post_content'));
    
    // Check if post parent and content are provided
    if (empty($post_parent) || empty($post_content)) {
        return new WP_Error('missing_data', 'Post parent and content are required.', array('status' => 400));
    }

    // Prepare the post data
    $post_data = array(
        'post_content' => $post_content,
        'post_type'    => 'reply', // Custom post type
        'post_status'  => 'publish', // Set the status of the post
        'post_parent'  => $post_parent, // Set the parent post ID
    );

    // Insert the post
    $post_id = wp_insert_post($post_data);

    // Check for errors
    if (is_wp_error($post_id)) {
        return $post_id; // Return the error
    }

    // Return success response
    return rest_ensure_response(array(
        'success' => true,
        'post_id' => $post_id,
    ));
}

// 
// END TOPIC REPLY
// 
// START PRODUCT API
add_action('rest_api_init', function () {
    register_rest_route('wc/v1', '/products', array(
        'methods' => 'GET',
        'callback' => 'get_products_data',
        'permission_callback' => '__return_true', // Change this to proper permissions in production
    ));
});

function get_products_data() {
    // Query WooCommerce products
 $args = array(
        'post_type' => 'product', // WooCommerce products post type
        'post_status' => 'publish', // Only get published products
        'numberposts' => -1, // Get all products
    );
    $products = get_posts($args);

    // Prepare the response array
    $response = array();

    foreach ($products as $product) {
        // Get product data
        $product_data = wc_get_product($product->ID); // WooCommerce function to get product object

        $product_info = array(
            'id' => $product->ID,
            'name' => $product->post_title,
            'price' => $product_data->get_price(),
            'stock_status' => $product_data->get_stock_status(),
            'sku' => $product_data->get_sku(),
            'date_created' => $product->post_date,
            'link' => get_permalink($product->ID),
            'survey_per_month' => get_field('survey_per_month', $product->ID), // Retrieve ACF field
        );

        $response[] = $product_info;
    }

    return new WP_REST_Response($response, 200);
}


// END PRODUCTS API
// 
// START SUBSCRIPTION API
// Add the REST API endpoint
add_action('rest_api_init', function () {
    register_rest_route('survey/v1', '/subscription-check', array(
        'methods' => 'POST',
        'callback' => 'check_survey_limit_and_subscription',
        'permission_callback' => '__return_true', // Adjust permissions as necessary
    ));
});

function check_survey_limit_and_subscription(WP_REST_Request $request) {
    global $wpdb;

    // Get the token from the request body
    $token = isset($request['token']) ? sanitize_text_field($request['token']) : '';

    // Check if the token is provided
    if (empty($token)) {
        return new WP_Error('missing_token', 'Token is required', array('status' => 403));
    }

    // Validate the token
    $user = get_users(array(
        'meta_key' => 'auth_token',
        'meta_value' => $token,
        'number' => 1,
        'fields' => 'ID'
    ));

    if (empty($user)) {
        return new WP_Error('invalid_token', 'Invalid or missing token', array('status' => 403));
    }
    $userid = $user[0];

    // Prepare the SQL query to check survey submissions in the last month
    $query = $wpdb->prepare('SELECT COUNT(*) as total FROM wp_survey WHERE user_id = %d AND created_at > NOW() - INTERVAL 1 MONTH', $userid);
    $result = $wpdb->get_results($query);
    $total = !empty($result) ? $result[0]->total : 0;

    // Get user's latest orders
    $orderArg = array(
        'customer_id' => $userid,
        'limit' => -1,
        'post_status' => array('processing', 'completed'),
        'orderby' => 'date',
        'order' => 'DESC',
    );

    $orders = wc_get_orders($orderArg);
    if (empty($orders)) {
        return new WP_Error('invalid_membership', 'Please buy a membership first', array('status' => 403));
    }

    $items = $orders[0]->get_items();

    // Get product attributes from the first order item
    foreach ($items as $item) {
        $pid = $item->get_product_id();
        break; // Get only the first item
    }

    $attributes = get_post_meta($pid);
    $number = isset($attributes['wps_sfw_subscription_number'][0]) ? $attributes['wps_sfw_subscription_number'][0] : 0;
    $interval = isset($attributes['wps_sfw_subscription_interval'][0]) ? $attributes['wps_sfw_subscription_interval'][0] : '';
    $spm = isset($attributes['survey_per_month'][0]) ? $attributes['survey_per_month'][0] : 0;

    // Calculate new expiration date
    $tdays = (1 * $number);
    $newdata = '';

    switch ($interval) {
        case 'year':
            $newdata = date('Y-m-d H:i:s', strtotime("+$tdays year"));
            break;
        case 'days':
            $newdata = date('Y-m-d H:i:s', strtotime("+$tdays days"));
            break;
        case 'weeks':
            $newdata = date('Y-m-d H:i:s', strtotime("+$tdays weeks"));
            break;
        case 'months':
            $newdata = date('Y-m-d H:i:s', strtotime("+$tdays months"));
            break;
    }

    $date_now = current_time('mysql', 1); // Get current time in MySQL format
    $mdate = strtotime($newdata);

    // Check membership status
    $etype = 0; // Default value
    if (strtotime($date_now) > $mdate) {
        $etype = 1; // Membership expired
    }
    if ($total >= $spm) {
        $etype = 2; // Per month limit exceeded
    }

    // Return the appropriate response based on the checks
    if ($etype == 2) {
        return new WP_REST_Response(['status' => 'error', 'message' => 'Your per month limit exceeded'], 403);
    }
    if ($etype == 1) {
        return new WP_REST_Response(['status' => 'error', 'message' => 'Your membership expired'], 403);
    }

    // If everything is fine, return a success message
    return new WP_REST_Response(['status' => 'success', 'message' => 'You can submit the survey'], 200);
}

// END SUBSCRIPTION API